package easik.ui.datatype;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;

import easik.Easik;
import easik.EasikConstants;
import easik.sketch.datatype.DataType;
import easik.sketch.datatype.DataTypeController;


/**
 * This class is used for the display of one single data type
 * within the data type controller ui.
 * 
 * @author Kevin Green 2006
 * @since 2006-06-21 Kevin Green
 * @version 2006-06-28 Kevin Green
 */
public class DataTypePane extends JPanel {

	/**
	 * The data type associated with the pane
	 */
	private DataType _dataType;
	/**
	 * The data type controller
	 */
	private DataTypeController _controller;
	/**
	 * The data type user interface
	 */
	private DatatypesUI _dataTypesUI;
	/**
	 * The remove button
	 */
	private JButton _removeButton;
	/**
	 * The edit button
	 */
	private JButton _editButton;
	/**
	 * The button pane
	 */
	private JPanel _buttonPane;
	/**
	 * The types pane
	 */
	private JPanel _typesPane;
	/**
	 * The data type information pane
	 */
	private JPanel _typeInfoPane;
	/**
	 * The User Defined platform pane
	 */
	private JPanel _UserDefinedPane;
	/**
	 * The MySQL platform pane
	 */
	private PlatformTypePane _MySQLPane;
	/**
	 * The Oracle platform pane
	 */
	private PlatformTypePane _OraclePane;
	/**
	 * The DB2 platform pane
	 */
	private PlatformTypePane _DB2Pane;
	/**
	 * The XML platform pane
	 */
	private PlatformTypePane _XMLPane;
	/**
	 * The data type name text box
	 */
	private JTextField _dataTypeNameBox;
	/**
	 * The data type description text box
	 */
	private JTextField _dataTypeDescBox;
	/**
	 * The User Defined platform text box
	 */
	private JTextField _userDefinedBox;
	/**
	 * True if the data type is editable, false otherwise
	 */
	private boolean _isEditable;
	
	/**
	 * Default Constructor.
	 * Makes the GUI components and sets it to be uneditiable by default.
	 * 
	 * @param inType The data type associated with the pane
	 * @param inCont The data type controller for the sketch
	 * @param inUI The parent GUI that this pane is displayed in
	 */
	public DataTypePane(DataType inType, DataTypeController inCont, DatatypesUI inUI){
		_dataType = inType;
		_controller = inCont;
		_dataTypesUI = inUI;
		
		//Layout the pane
		this.setLayout(new BorderLayout());
		this.setBorder(new CompoundBorder(new EmptyBorder(1,1,4,1), new LineBorder(Color.BLACK, 1)));
		
		//Create and then disable the GUI
		makePane();
		setEditability(false);
	}
	
	/**
	 * Makes the main pane
	 */
	public void makePane(){
		//Create panes
		makeButtonPane();
		makeTypeInfoPane();
		_MySQLPane = new PlatformTypePane(EasikConstants.MY_SQL_TYPES, EasikConstants.MY_SQL_DEFAULT, _dataType.get_MySQL_type());
		_OraclePane = new PlatformTypePane(EasikConstants.ORACLE_TYPES, EasikConstants.MY_SQL_DEFAULT, _dataType.get_Oracle_type());
		_DB2Pane = new PlatformTypePane(EasikConstants.DB2_TYPES, EasikConstants.MY_SQL_DEFAULT, _dataType.get_DB2_type());
		_XMLPane = new PlatformTypePane(EasikConstants.XML_TYPES, EasikConstants.MY_SQL_DEFAULT, _dataType.get_XML_type());
		makeUserDefinedPane();
		
		//Add spacing to the panes
		Border myBorder = new EmptyBorder(1,1,1,1);
		_buttonPane.setBorder(myBorder);
		_typeInfoPane.setBorder(myBorder);
		_MySQLPane.setBorder(myBorder);
		_OraclePane.setBorder(myBorder);
		_DB2Pane.setBorder(myBorder);
		_XMLPane.setBorder(myBorder);
		_UserDefinedPane.setBorder(myBorder);
		
		//Hides elements that are not currently used
		setTypeVisibility();
		
		//Add panes to self
		_typesPane = new JPanel(new GridLayout(1, 0));
		this.add(_buttonPane, BorderLayout.WEST);
		this.add(_typesPane, BorderLayout.CENTER);
		_typesPane.add(_typeInfoPane);
		if(_controller.is_useMySQL())
			_typesPane.add(_MySQLPane);
		if(_controller.is_useOracle())
			_typesPane.add(_OraclePane);
		if(_controller.is_useDB2())
			_typesPane.add(_DB2Pane);
		if(_controller.is_useXML())
			_typesPane.add(_XMLPane);
		if(_controller.is_useUserDefined())
			_typesPane.add(_UserDefinedPane);
	}
	
	/**
	 * Makes the button pane
	 */
	public void makeButtonPane(){
		_buttonPane = new JPanel(new GridLayout(2,1));
		_removeButton = new JButton("Remove");
		_editButton = new JButton("Edit");
		
		//Add buttons to the pane
		//_buttonPane.add(new JLabel(" "));
		_buttonPane.add(_removeButton);
		_buttonPane.add(_editButton);
		//_buttonPane.add(new JLabel(" "));

		//Add action listeners to the buttons
		_removeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				removeDataType();
			}
		});
		_editButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				setEditability(!_isEditable);
			}
		});
	}
	
	/**
	 * Makes the data type information pane
	 */
	public void makeTypeInfoPane(){
		_typeInfoPane = new JPanel();
		
		_typeInfoPane.setLayout(new GridLayout(3,1));
		
		//Add components for name
		_dataTypeNameBox = new JTextField();
		_dataTypeNameBox.setFont(_dataTypeNameBox.getFont().deriveFont(Font.BOLD));
		_typeInfoPane.add(_dataTypeNameBox);
		_dataTypeNameBox.setText(_dataType.getTypeName());

		//Add components for description
		JLabel typeDescLabel = new JLabel("Type Desc:");
		typeDescLabel.setFont(typeDescLabel.getFont().deriveFont(Font.PLAIN));
		_typeInfoPane.add(typeDescLabel);
		_dataTypeDescBox = new JTextField();
		_dataTypeDescBox.setFont(_dataTypeDescBox.getFont().deriveFont(Font.ITALIC));
		_typeInfoPane.add(_dataTypeDescBox);
		_dataTypeDescBox.setText(_dataType.getTypeDesc());
	}
	
	/**
	 * Makes the user defined data type pane
	 */
	public void makeUserDefinedPane(){
		_UserDefinedPane = new JPanel();
		
		_UserDefinedPane.setLayout(new GridLayout(3,1));
		
		//Add user type components
		_userDefinedBox = new JTextField();
		_UserDefinedPane.add(_userDefinedBox);
		_userDefinedBox.setText(_dataType.get_user_type());
	}
	
	/**
	 * Sets only the types that active to visible.
	 */
	public void setTypeVisibility(){
		if(_controller.is_useMySQL())
			_MySQLPane.setVisible(true);
		else
			_MySQLPane.setVisible(false);
		
		if(_controller.is_useOracle())
			_OraclePane.setVisible(true);
		else
			_OraclePane.setVisible(false);
		
		if(_controller.is_useDB2())
			_DB2Pane.setVisible(true);
		else
			_DB2Pane.setVisible(false);
		
		if(_controller.is_useXML())
			_XMLPane.setVisible(true);
		else
			_XMLPane.setVisible(false);
		
		if(_controller.is_useUserDefined())
			_UserDefinedPane.setVisible(true);
		else
			_UserDefinedPane.setVisible(false);
	}
	
	/**
	 * Removes the data type associated with this pane from
	 * the data type controller and this pane from the window.
	 */
	public void removeDataType(){
		if(_controller.getDataTypes().size()>1){
			//Get replacement
			Object[] typeArr = Easik.getInstance().getFrame().getSketch().getDataTypeController().getDataTypes().toArray();
			DataType replacementType = 
				(DataType) JOptionPane.showInputDialog(
                    this.getParent(),
                    "Please select the data type to replace the '" + _dataType.getTypeName() + "' data type with.",
                    "Replacement Data Type",
                    JOptionPane.PLAIN_MESSAGE,
                    null,
                    typeArr,
                    typeArr[0]);
			
			//Check replacement selection
			if(replacementType != null && replacementType != _dataType){ //If replacement was selected
				_controller.removeDataType(_dataType, replacementType);
				_dataTypesUI.removeDataTypePane(this);
			}
			else{//No replacement selected, do nothing
				return;
			}
		}
		else{
			JOptionPane.showMessageDialog(
					this.getParent(),
					"Remove failed!\nThere must always be at least one data type.",
					"Error",
					JOptionPane.ERROR_MESSAGE);
		}
	}
	
	/**
	 * Sets if the data type is editable or disabled from changes
	 * 
	 * @param inVal True is editable, false if disabled from changes.
	 */
	public void setEditability(boolean inVal){
		if(inVal){
			_editButton.setText("Save");
			_dataTypeNameBox.grabFocus();
			_dataTypeNameBox.selectAll();
		}
		else{
			if(saveErrorCheck()){
				saveValues();
				_editButton.setText("Edit");
			}
			else{
				return;
			}
		}
		
		_isEditable = inVal;
		((PlatformTypePane)_MySQLPane).setEditability(inVal);
		((PlatformTypePane)_OraclePane).setEditability(inVal);
		((PlatformTypePane)_DB2Pane).setEditability(inVal);
		((PlatformTypePane)_XMLPane).setEditability(inVal);
		_dataTypeNameBox.setEditable(inVal);
		_dataTypeDescBox.setEditable(inVal);
		_userDefinedBox.setEditable(inVal);
	}
	
	/**
	 * Error checks before the save operation.
	 * 
	 * @return True if there are no errors, false if there are errors.
	 */
	public boolean saveErrorCheck(){
		if(!(_dataTypeNameBox.getText().equals("") || 
				(_MySQLPane.getDataType().equals("")) ||
				(_OraclePane.getDataType().equals("")) ||
				(_DB2Pane.getDataType().equals("")) ||
				(_XMLPane.getDataType().equals("")) ||
				(_userDefinedBox.getText().equals("")))){
			if(_controller.isNameFree(_dataType, _dataTypeNameBox.getText())){
				saveValues();
				return true;
			}
			else{
				JOptionPane.showMessageDialog(
						this.getParent(),
						"A duplicate name was detected while trying to save the " + _dataTypeNameBox.getText() + " data type.\nPlease rename and try to save again!",
						"Error",
						JOptionPane.ERROR_MESSAGE);
				return false;
			}
		}
		else{
			JOptionPane.showMessageDialog(
					this.getParent(),
					"An error has occured while trying to save the " + _dataTypeNameBox.getText() + " data type.",
					"Error",
					JOptionPane.ERROR_MESSAGE);
			return false;
		}
	}
	
	/**
	 * Saves the values from the GUI back into the data type
	 */
	private void saveValues(){
		_dataType.setTypeName(_dataTypeNameBox.getText());
		_dataType.setTypeDesc(_dataTypeDescBox.getText());
		_dataType.set_MySQL_type(_MySQLPane.getDataType());
		_dataType.set_Oracle_type(_OraclePane.getDataType());
		_dataType.set_DB2_type(_DB2Pane.getDataType());
		_dataType.set_XML_type(_XMLPane.getDataType());
		_dataType.set_user_type(_userDefinedBox.getText());
	}

	/**
	 * @return Returns the _isEditable.
	 */
	public boolean isEditable() {
		return _isEditable;
	}	
}
